package ga.view.streaming.showroom;

import math.geom2d.Point2D;
import math.geom2d.line.LineSegment2D;
import math.geom2d.polygon.Polygon2D;
import math.geom2d.polygon.Rectangle2D;

import com.jme3.asset.AssetManager;
import com.jme3.light.AmbientLight;
import com.jme3.light.PointLight;
import com.jme3.material.Material;
import com.jme3.math.ColorRGBA;
import com.jme3.math.FastMath;
import com.jme3.math.Vector2f;
import com.jme3.math.Vector3f;
import com.jme3.renderer.queue.RenderQueue.ShadowMode;
import com.jme3.scene.Geometry;
import com.jme3.scene.Mesh;
import com.jme3.scene.Node;
import com.jme3.scene.shape.Box;
import com.jme3.system.AppSettings;
import com.jme3.util.TangentBinormalGenerator;

/**
 * This is a {@link ShowRoom} that has a rectangular floor area. With the walls
 * it is a box.
 * 
 * @since 12.08.2012
 * @author Stephan Dreyer
 */
public final class BoxShowRoom extends ShowRoom {

  private final Node walls;
  private final Geometry floor;
  private final Geometry roof;

  private final Rectangle2D bounds;

  /**
   * Enum of the types of lighting.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public enum LightingType {
    /** Warm ambient lighting from multiple angles. */
    AMBIENT,
    /** Warm atmospheric lighting. */
    ATMOSPHERE,
    /** Cold white lighting as in an office. */
    OFFICE
  }

  /**
   * Instantiates a new box show room.
   * 
   * @param assetManager
   *          the asset manager
   * @param settings
   *          the settings
   * @param showRoomSettings
   *          the show room settings
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public BoxShowRoom(final AssetManager assetManager,
      final AppSettings settings, final ShowRoomSettings showRoomSettings) {
    super("Show Room", assetManager, settings, showRoomSettings);

    final float roomWidth = showRoomSettings.getFloat(
        ShowRoomSettings.BOX_WIDTH, 5f);
    final float roomLength = showRoomSettings.getFloat(
        ShowRoomSettings.BOX_LENGTH, 5f);
    final float roomHeight = showRoomSettings.getFloat(
        ShowRoomSettings.BOX_HEIGHT, 2.8f);

    final float wallThickness = 0.1f;

    final float roomWidthHalf = roomWidth / 2f;
    final float roomLengthHalf = roomLength / 2f;

    walls = new Node("walls");

    Mesh m = new Wall(new Vector3f(roomWidthHalf, 0f, -roomLengthHalf),
        new Vector3f(roomWidthHalf + wallThickness, roomHeight, roomLengthHalf));

    Geometry geo = new Geometry("wall1", m);
    geo.getLocalRotation().fromAngles(0f, FastMath.DEG_TO_RAD * 180f, 0f);
    geo.setMaterial(wallMaterial);
    walls.attachChild(geo);

    m = new Wall(new Vector3f(roomLengthHalf, 0f, -roomWidthHalf),
        new Vector3f(roomLengthHalf + wallThickness, roomHeight, roomWidthHalf));
    geo = new Geometry("wall2", m);
    geo.setMaterial(wallMaterial);
    geo.getLocalRotation().fromAngles(0f, FastMath.DEG_TO_RAD * 270f, 0f);
    walls.attachChild(geo);

    m = new Wall(new Vector3f(roomWidthHalf, 0f, -roomLengthHalf),
        new Vector3f(roomWidthHalf + wallThickness, roomHeight, roomLengthHalf));
    geo = new Geometry("wall3", m);
    geo.setMaterial(wallMaterial);
    walls.attachChild(geo);

    m = new Wall(new Vector3f(roomLengthHalf, 0f, -roomWidthHalf),
        new Vector3f(roomLengthHalf + wallThickness, roomHeight, roomWidthHalf));

    geo = new Geometry("wall4", m);
    geo.setMaterial(wallMaterial);
    geo.getLocalRotation().fromAngles(0f, FastMath.DEG_TO_RAD * 90f, 0f);
    walls.attachChild(geo);

    walls.setCullHint(CullHint.Dynamic);
    // walls.setQueueBucket(Bucket.Transparent);

    attachChild(walls);

    floor = new Geometry("floor", new Box(new Vector3f(roomWidthHalf, 0f,
        roomLengthHalf), new Vector3f(-roomWidthHalf, 0f, -roomLengthHalf)));
    floor.setMaterial(floorMaterial);

    final float floorScale = 2;

    floor.getMesh().scaleTextureCoordinates(
        new Vector2f(roomLength / floorScale, roomWidth / floorScale));
    attachChild(floor);

    m = new Wall(new Vector3f(0, -roomWidthHalf, -roomLengthHalf),
        new Vector3f(wallThickness, roomWidthHalf, roomLengthHalf));
    roof = new Geometry("ceiling", m);
    roof.setMaterial(ceilingMaterial);
    roof.getLocalRotation().fromAngles(0f, 0f, FastMath.DEG_TO_RAD * 90f);
    roof.setLocalTranslation(0f, roomHeight, 0f);
    attachChild(roof);

    updateModelBound();

    setShadowMode(ShadowMode.Receive);

    TangentBinormalGenerator.generate(this);

    bounds = new Rectangle2D(-roomWidthHalf, -roomLengthHalf, roomWidth,
        roomLength);

    createLights();

    updateGeometricState();
  }

  /**
   * Creates the lights.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private void createLights() {

    final float roomWidth = showRoomSettings.getFloat(
        ShowRoomSettings.BOX_WIDTH, 5f);
    final float roomLength = showRoomSettings.getFloat(
        ShowRoomSettings.BOX_LENGTH, 5f);
    final float roomHeight = showRoomSettings.getFloat(
        ShowRoomSettings.BOX_HEIGHT, 2.8f);

    final float roomWidthHalf = roomWidth / 2f;
    final float roomLengthHalf = roomLength / 2f;

    LightingType type = showRoomSettings.get(ShowRoomSettings.LIGHTING_TYPE);

    if (type == null) {
      type = LightingType.AMBIENT;
    }

    Vector3f pos;
    PointLight pl;

    ColorRGBA lightC;
    float radius;

    switch (type) {
    case OFFICE:

      lightC = new ColorRGBA(0.55f, 0.55f, 0.6f, 0.2f);

      radius = (int) (Math.min(roomWidth, roomLength) * 1.2f);

      final int lightCount = Math.min(
          (int) (Math.max(roomWidth, roomLength) * 0.6f), 8);

      final float lightDistance = 1.6f;

      // / LIGHTS
      for (int i = 1; i <= lightCount; i++) {

        float x = Math.min(roomWidthHalf, roomLengthHalf) * 0.9f;
        final float z = lightDistance * (((lightCount + 1) * 0.5f) - i);

        pl = new PointLight();

        pos = new Vector3f(roomLength < roomWidth ? z : x, roomHeight * 0.8f,
            roomLength < roomWidth ? x : z);

        pl.setPosition(pos);
        pl.setColor(lightC);
        pl.setRadius(radius);

        addLight(pl);

        createLightObj(pos);

        x = -x;

        pl = new PointLight();

        pos = new Vector3f(roomLength < roomWidth ? z : x, roomHeight * 0.8f,
            roomLength < roomWidth ? x : z);

        pl.setPosition(pos);
        pl.setColor(lightC);
        pl.setRadius(radius);

        addLight(pl);

        createLightObj(pos);
      }

      addLight(new AmbientLight());

      break;

    case ATMOSPHERE:

      lightC = new ColorRGBA(.8f, .8f, 0.6f, 0.8f);

      pos = new Vector3f(roomWidthHalf * 0.8f, roomHeight * 0.8f,
          roomLengthHalf * 0.8f);

      // / LIGHTS
      pl = new PointLight();
      pl.setPosition(pos);
      pl.setColor(lightC);
      pl.setRadius(roomHeight * 2f);

      addLight(pl);
      createLightObj(pos);

      pos = new Vector3f(-roomWidthHalf * 0.8f, roomHeight * 0.8f,
          -roomLengthHalf * 0.8f);

      pl = new PointLight();
      pl.setPosition(pos);
      pl.setColor(lightC);
      pl.setRadius(roomHeight * 2f);

      addLight(pl);
      createLightObj(pos);

      pos = new Vector3f(roomWidthHalf * 0.8f, roomHeight * 0.8f,
          -roomLengthHalf * 0.8f);

      pl = new PointLight();
      pl.setPosition(pos);
      pl.setColor(lightC);
      pl.setRadius(roomHeight * 2f);

      addLight(pl);
      createLightObj(pos);

      pos = new Vector3f(-roomWidthHalf * 0.8f, roomHeight * 0.8f,
          roomLengthHalf * 0.8f);

      pl = new PointLight();
      pl.setPosition(pos);
      pl.setColor(lightC);
      pl.setRadius(roomHeight * 2f);

      addLight(pl);
      createLightObj(pos);

      addLight(new AmbientLight());

      break;

    case AMBIENT:
    default:

      lightC = new ColorRGBA(.8f, .8f, 0.6f, 0.8f);

      radius = Math.max(roomWidth, roomLength);

      final float wallDist = 0.4f;

      pos = new Vector3f(roomWidthHalf - wallDist, roomHeight * 0.9f, 0f);

      pl = new PointLight();
      pl.setPosition(pos);
      pl.setColor(lightC);
      pl.setRadius(radius);

      addLight(pl);
      createLightObj(pos);

      pos = new Vector3f(-roomWidthHalf + wallDist, roomHeight * 0.9f, 0f);

      pl = new PointLight();
      pl.setPosition(pos);
      pl.setColor(lightC);
      pl.setRadius(radius);

      addLight(pl);
      createLightObj(pos);

      pos = new Vector3f(0f, roomHeight * 0.9f, roomLengthHalf - wallDist);

      pl = new PointLight();
      pl.setPosition(pos);
      pl.setColor(lightC);
      pl.setRadius(radius);

      addLight(pl);
      createLightObj(pos);

      pos = new Vector3f(0f, roomHeight * 0.9f, -roomLengthHalf + wallDist);

      pl = new PointLight();
      pl.setPosition(pos);
      pl.setColor(lightC);
      pl.setRadius(radius);

      addLight(pl);
      createLightObj(pos);

      addLight(new AmbientLight());

      break;
    }
  }

  /**
   * Method for debugging. Attaches a cylinder for every light to show the light
   * origins.
   * 
   * @param pos
   *          The position of the light.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private void createLightObj(final Vector3f pos) {
    // final Mesh m = new Cylinder(6, 6, 0.1f, 0.2f, true);
    //
    // final Geometry geo = new Geometry("Object", m);
    // geo.rotateUpTo(Vector3f.UNIT_Z);
    // geo.setMaterial(wallMaterial);
    // geo.setShadowMode(ShadowMode.CastAndReceive);
    // geo.setQueueBucket(Bucket.Transparent);
    //
    // attachChild(geo);
    // geo.setLocalTranslation(pos);
  }

  /**
   * Gets the material.
   * 
   * @deprecated This method is only for debugging and will be removed soon.
   * @return the material
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  @Deprecated
  public Material getMaterial() {
    return wallMaterial;
  }

  @Override
  public boolean contains(final Polygon2D shape) {
    return bounds.getBoundingBox().containsBounds(shape);
  }

  @Override
  public float minDistanceToWall(final Polygon2D shape, final float angle) {
    float minDistance = Float.MAX_VALUE;

    // iterate through edges of the room
    for (final LineSegment2D edge : bounds.getEdges()) {

      // iterate through vertices of the bounding shape
      for (final Point2D vert : shape.getVertices()) {

        boolean sameDir = false;

        // this works ok, but assumes that only the back side is allowed to face
        // the wall

        if (Math.abs(angle) < 0.0001f) {
          // assume no rotation

          if (edge.getFirstPoint().getY() < vert.getY()
              && edge.getLastPoint().getY() < vert.getY()) {
            sameDir = true;
          }
        } else if (Math.abs(angle - (FastMath.DEG_TO_RAD * 90f)) < 0.0001f) {
          // assume 90 degrees

          if (edge.getFirstPoint().getX() < vert.getX()
              && edge.getLastPoint().getX() < vert.getX()) {
            sameDir = true;
          }
        } else if (Math.abs(angle - (FastMath.DEG_TO_RAD * 180f)) < 0.0001f) {

          if (edge.getFirstPoint().getY() > vert.getY()
              && edge.getLastPoint().getY() > vert.getY()) {
            sameDir = true;
          }
        } else if (Math.abs(angle - (FastMath.DEG_TO_RAD * 270f)) < 0.0001f) {

          if (edge.getFirstPoint().getX() > vert.getX()
              && edge.getLastPoint().getX() > vert.getX()) {
            sameDir = true;
          }
        }

        if (sameDir) {
          // get the min distance between all vertices and edges
          minDistance = Math.min(minDistance, (float) edge.getDistance(vert));
        }
      }
    }

    return minDistance;
  }

  @Override
  public Rectangle2D getOutterBounds() {
    return bounds;
  }
}
